<?php
    
    namespace artbox\weblog\controllers;

    use artbox\weblog\models\Category;
    use artbox\weblog\models\Tag;
    use Yii;
    use artbox\weblog\models\Article;
    use artbox\weblog\models\ArticleSearch;
    use yii\helpers\ArrayHelper;
    use yii\web\Controller;
    use yii\web\NotFoundHttpException;
    use yii\filters\VerbFilter;
    use yii\web\Response;
    use yii\filters\AccessControl;
    
    /**
     * BlogArticleController implements the CRUD actions for BlogArticle model.
     */
    class ArticleController extends Controller
    {
        /**
         * @inheritdoc
         */
        public function getViewPath()
        {
            return '@artbox/weblog/views/blog-article';
        }
        
        /**
         * @inheritdoc
         */
        public function behaviors()
        {
            return [
                'verbs' => [
                    'class'   => VerbFilter::className(),
                    'actions' => [
                        'delete' => [ 'POST' ],
                    ],
                ],
                'access' => [
                    'class' => AccessControl::className(),
                    'rules' => [
                        [
                            'actions' => [
                                'login',
                                'error',
                            ],
                            'allow'   => true,
                        ],
                        [
                            'allow' => true,
                            'roles' => [ '@' ],
                        ],
                    ],
                ],
            ];
        }
        
        /**
         * Lists all BlogArticle models.
         *
         * @return mixed
         */
        public function actionIndex()
        {
            $searchModel = new ArticleSearch();
            $dataProvider = $searchModel->search(Yii::$app->request->queryParams);
            
            return $this->render(
                'index',
                [
                    'searchModel'  => $searchModel,
                    'dataProvider' => $dataProvider,
                ]
            );
        }
        
        /**
         * Displays a single BlogArticle model.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionView($id)
        {
            return $this->render(
                'view',
                [
                    'model' => $this->findModel($id),
                ]
            );
        }
        
        /**
         * Creates a new BlogArticle model.
         * If creation is successful, the browser will be redirected to the 'view' page.
         *
         * @return mixed
         */
        public function actionCreate()
        {
            $model = new Article();
            $model->generateLangs();
    
            if (class_exists('\artbox\catalog\models\Product')) {
                $model->productIds = ArrayHelper::map(
                    $model->relatedProducts,
                    'id',
                    'lang.title'
                );
            }
    
            if ($model->loadWithLangs(\Yii::$app->request) && $model->saveWithLangs()) {
        
                $categories = Category::find()
                                      ->where([ 'id' => \Yii::$app->request->post('categoryIds') ])
                                      ->all();
    
                if (class_exists('\artbox\catalog\models\Product')) {
                    /**
                     * @var \yii\db\ActiveQuery $query
                     */
                    $query = call_user_func(
                        [
                            '\artbox\catalog\models\Product',
                            'find',
                        ]
                    );
                    /**
                     * @var \artbox\catalog\models\Product[] $products
                     */
                    $products = $query->where([ 'id' => \Yii::$app->request->post('productIds') ])
                                      ->all();
        
                    $model->linkMany('relatedProducts', $products);
                }
                
        
                $model->linkMany('categories', $categories);
        
                $tags = Tag::find()
                           ->where(
                               [
                                   'id' => \Yii::$app->request->post('tagIds'),
                               ]
                           )
                           ->all();
        
                $model->linkMany('tags', $tags);
        
                return $this->redirect(
                    [
                        'view',
                        'id' => $model->id,
                    ]
                );
            }
            
            return $this->render(
                'create',
                [
                    'model'      => $model,
                    'modelLangs' => $model->modelLangs,
                ]
            );
            
        }
        
        /**
         * Updates an existing BlogArticle model.
         * If update is successful, the browser will be redirected to the 'view' page.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionUpdate($id)
        {
            $model = $this->findModel($id);
            $model->generateLangs();
    
            $model->categoryIds = ArrayHelper::map(
                $model->categories,
                'id',
                'lang.title'
            );
    
            $model->tagIds = ArrayHelper::map(
                $model->tags,
                'id',
                'lang.label'
            );
    
            $model->articleIds = ArrayHelper::map(
                $model->articles,
                'id',
                'lang.title'
            );
    
            if (class_exists('\artbox\catalog\models\Product')) {
                $model->productIds = ArrayHelper::map(
                    $model->relatedProducts,
                    'id',
                    'lang.title'
                );
            }
            
            if ($model->loadWithLangs(\Yii::$app->request) && $model->saveWithLangs()) {
                $categories = Category::find()
                                      ->where([ 'id' => \Yii::$app->request->post('categoryIds') ])
                                      ->all();
    
                $model->linkMany('categories', $categories);
    
                $tags = Tag::find()
                           ->where(
                               [
                                   'id' => \Yii::$app->request->post('tagIds'),
                               ]
                           )
                           ->all();
    
                $model->linkMany('tags', $tags);
    
                if (class_exists('\artbox\catalog\models\Product')) {
                    /**
                     * @var \yii\db\ActiveQuery $query
                     */
                    $query = call_user_func(
                        [
                            '\artbox\catalog\models\Product',
                            'find',
                        ]
                    );
                    /**
                     * @var \artbox\catalog\models\Product[] $products
                     */
                    $products = $query->where([ 'id' => \Yii::$app->request->post('productIds') ])
                                      ->all();
        
                    $model->linkMany('relatedProducts', $products);
                }
                
                return $this->redirect(
                    [
                        'view',
                        'id' => $model->id,
                    ]
                );
                
            }
            return $this->render(
                'update',
                [
                    'model'      => $model,
                    'modelLangs' => $model->modelLangs,
                ]
            );
            
        }
        
        /**
         * Deletes an existing BlogArticle model.
         * If deletion is successful, the browser will be redirected to the 'index' page.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionDelete($id)
        {
            $this->findModel($id)
                 ->delete();
            
            return $this->redirect([ 'index' ]);
        }
        
        /**
         * Finds the BlogArticle model based on its primary key value.
         * If the model is not found, a 404 HTTP exception will be thrown.
         *
         * @param integer $id
         *
         * @return Article the loaded model
         * @throws NotFoundHttpException if the model cannot be found
         */
        protected function findModel($id)
        {
            if (( $model = Article::findOne($id) ) !== null) {
                return $model;
            } else {
                throw new NotFoundHttpException('The requested page does not exist.');
            }
        }
        
        /**
         * @param string  $q
         * @param integer $id
         *
         * @return array
         */
        public function actionList(string $q = null, int $id = null)
        {
            \Yii::$app->response->format = Response::FORMAT_JSON;
            $out = [
                'results' => [
                    'id'   => '',
                    'text' => '',
                ],
            ];
            if (!is_null($q)) {
                $out[ 'results' ] = Article::find()
                                           ->joinWith('lang')
                                           ->select(
                                               [
                                                   'blog_article.id as id',
                                                   'blog_article_lang.title as text',
                                               ]
                                           )
                                           ->where(
                                               [
                                                   'like',
                                                   'blog_article_lang.title',
                                                   $q,
                                               ]
                                           )
                                           ->andFilterWhere(
                                               [
                                                   '!=',
                                                   'blog_article.id',
                                                   $id,
                                               ]
                                           )
                                           ->limit(20)
                                           ->asArray()
                                           ->all();
            }
            return $out;
        }
    }
