<?php
    
    namespace artbox\weblog\controllers;

    use artbox\weblog\models\Category;
    use artbox\weblog\models\CategorySearch;
    use Yii;
    use yii\helpers\ArrayHelper;
    use yii\web\Controller;
    use yii\web\NotFoundHttpException;
    use yii\filters\VerbFilter;
    use yii\web\Response;
    use yii\filters\AccessControl;

    /**
     * BlogCategoryController implements the CRUD actions for BlogCategory model.
     */
    class CategoryController extends Controller
    {
        /**
         * @inheritdoc
         */
        public function getViewPath()
        {
            return '@artbox/weblog/views/blog-category';
        }
    
        /**
         * @inheritdoc
         */
        public function behaviors()
        {
            return [
                'verbs' => [
                    'class'   => VerbFilter::className(),
                    'actions' => [
                        'delete' => [ 'POST' ],
                    ],
                ],
                'access' => [
                    'class' => AccessControl::className(),
                    'rules' => [
                        [
                            'actions' => [
                                'login',
                                'error',
                            ],
                            'allow'   => true,
                        ],
                        [
                            'allow' => true,
                            'roles' => [ '@' ],
                        ],
                    ],
                ],
            ];
        }
    
        /**
         * Lists all BlogCategory models.
         *
         * @return mixed
         */
        public function actionIndex()
        {
            $searchModel = new CategorySearch();
            $dataProvider = $searchModel->search(Yii::$app->request->queryParams);
    
            return $this->render(
                'index',
                [
                    'searchModel'  => $searchModel,
                    'dataProvider' => $dataProvider,
                ]
            );
        }
    
        /**
         * Displays a single BlogCategory model.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionView($id)
        {
            return $this->render(
                'view',
                [
                    'model' => $this->findModel($id),
                ]
            );
        }
    
        /**
         * Creates a new BlogCategory model.
         * If creation is successful, the browser will be redirected to the 'view' page.
         *
         * @return mixed
         */
        public function actionCreate()
        {
            $model = new Category();
            $model->generateLangs();
            $parentCategories = ArrayHelper::map(
                Category::find()
                        ->joinWith('lang')
                        ->where(
                            [
                                'parent_id' => 0,
                            ]
                        )
                        ->all(),
                'id',
                'lang.title'
            );
    
            if ($model->loadWithLangs(\Yii::$app->request) && $model->saveWithLangs()) {
                return $this->redirect(
                    [
                        'view',
                        'id' => $model->id,
                    ]
                );
            }
            return $this->render(
                'create',
                [
                    'model'            => $model,
                    'modelLangs'       => $model->modelLangs,
                    'parentCategories' => $parentCategories,
                ]
            );
    
        }
    
        /**
         * Updates an existing BlogCategory model.
         * If update is successful, the browser will be redirected to the 'view' page.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionUpdate($id)
        {
            $model = $this->findModel($id);
            $model->generateLangs();
            $parentCategories = ArrayHelper::map(
                Category::find()
                        ->joinWith('lang')
                        ->where(
                            [
                                'parent_id' => 0,
                            ]
                        )
                        ->andWhere(
                            [
                                '!=',
                                Category::tableName() . '_id',
                                $model->id,
                            ]
                        )
                        ->all(),
                'id',
                'lang.title'
            );
            if ($model->loadWithLangs(\Yii::$app->request) && $model->saveWithLangs()) {
                return $this->redirect(
                    [
                        'view',
                        'id' => $model->id,
                    ]
                );
                
            }
            return $this->render(
                'update',
                [
                    'model'            => $model,
                    'modelLangs'       => $model->modelLangs,
                    'parentCategories' => $parentCategories,
                ]
            );
    
        }
    
        /**
         * Deletes an existing BlogCategory model.
         * If deletion is successful, the browser will be redirected to the 'index' page.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionDelete($id)
        {
            $this->findModel($id)
                 ->delete();
    
            return $this->redirect([ 'index' ]);
        }
    
        public function actionList(string $q = null)
        {
            \Yii::$app->response->format = Response::FORMAT_JSON;
            $out = [
                'results' => [
                    [
                        'id'   => '',
                        'text' => '',
                    ],
                ],
            ];
            if (!is_null($q)) {
                $categories = Category::find()
                                      ->joinWith('lang')
                                      ->select(
                                          [
                                              'id'   => 'blog_category.id',
                                              'text' => 'blog_category_lang.title',
                                          ]
                                      )
                                      ->andFilterWhere(
                                          [
                                              'like',
                                              'blog_category_lang.title',
                                              $q,
                                          ]
                                      )
                                      ->limit(20)
                                      ->asArray()
                                      ->all();
        
                if (!empty($categories)) {
                    $out[ 'results' ] = $categories;
                }
            }
    
            return $out;
        }
    
        /**
         * Finds the BlogCategory model based on its primary key value.
         * If the model is not found, a 404 HTTP exception will be thrown.
         
         *
*@param integer $id
         *
         * @return Category the loaded model
         * @throws NotFoundHttpException if the model cannot be found
         */
        protected function findModel($id)
        {
            if (( $model = Category::findOne($id) ) !== null) {
                return $model;
            } else {
                throw new NotFoundHttpException('The requested page does not exist.');
            }
        }
    }
