<?php
    
    namespace artbox\weblog\controllers;

    use artbox\weblog\models\Tag;
    use artbox\weblog\models\TagSearch;
    use Yii;
    use yii\web\Controller;
    use yii\web\NotFoundHttpException;
    use yii\filters\VerbFilter;
    use yii\web\Response;
    use yii\filters\AccessControl;

    /**
     * BlogTagController implements the CRUD actions for BlogTag model.
     */
    class TagController extends Controller
    {
        public function getViewPath()
        {
            return '@artbox/weblog/views/blog-tag';
        }
    
        /**
         * @inheritdoc
         */
        public function behaviors()
        {
            return [
                'verbs' => [
                    'class'   => VerbFilter::className(),
                    'actions' => [
                        'delete' => [ 'POST' ],
                    ],
                ],
                'access' => [
                    'class' => AccessControl::className(),
                    'rules' => [
                        [
                            'actions' => [
                                'login',
                                'error',
                            ],
                            'allow'   => true,
                        ],
                        [
                            'allow' => true,
                            'roles' => [ '@' ],
                        ],
                    ],
                ],
            ];
        }
    
        /**
         * Lists all BlogTag models.
         *
         * @return mixed
         */
        public function actionIndex()
        {
            $searchModel = new TagSearch();
            $dataProvider = $searchModel->search(Yii::$app->request->queryParams);
        
            return $this->render(
                'index',
                [
                    'searchModel'  => $searchModel,
                    'dataProvider' => $dataProvider,
                ]
            );
        }
    
        /**
         * Displays a single BlogTag model.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionView($id)
        {
            return $this->render(
                'view',
                [
                    'model' => $this->findModel($id),
                ]
            );
        }
    
        /**
         * Creates a new BlogTag model.
         * If creation is successful, the browser will be redirected to the 'view' page.
         *
         * @return mixed
         */
        public function actionCreate()
        {
            $model = new Tag();
            $model->generateLangs();
        
            if (\Yii::$app->request->isPost) {
                $model->loadLangs(\Yii::$app->request);
                if ($model->saveWithLangs()) {
                    return $this->redirect(
                        [
                            'view',
                            'id' => $model->id,
                        ]
                    );
                }
            }
            return $this->render(
                'create',
                [
                    'model'      => $model,
                    'modelLangs' => $model->modelLangs,
                ]
            );
        
        }
    
        /**
         * Updates an existing BlogTag model.
         * If update is successful, the browser will be redirected to the 'view' page.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionUpdate($id)
        {
            $model = $this->findModel($id);
            $model->generateLangs();
        
            if (Yii::$app->request->isPost) {
                $model->loadLangs(\Yii::$app->request);
                if ($model->saveWithLangs()) {
                    return $this->redirect(
                        [
                            'view',
                            'id' => $model->id,
                        ]
                    );
                }
            }
            return $this->render(
                'update',
                [
                    'model'      => $model,
                    'modelLangs' => $model->modelLangs,
                ]
            );
        
        }
    
        /**
         * Deletes an existing BlogTag model.
         * If deletion is successful, the browser will be redirected to the 'index' page.
         *
         * @param integer $id
         *
         * @return mixed
         */
        public function actionDelete($id)
        {
            $this->findModel($id)
                 ->delete();
        
            return $this->redirect([ 'index' ]);
        }
    
        public function actionList(string $q = null)
        {
            \Yii::$app->response->format = Response::FORMAT_JSON;
            $out = [
                'results' => [
                    [
                        'id'   => '',
                        'text' => '',
                    ],
                ],
            ];
            if (!is_null($q)) {
                $categories = Tag::find()
                                 ->joinWith('lang')
                                 ->select(
                                     [
                                         'id'   => 'blog_tag.id',
                                         'text' => 'blog_tag_lang.label',
                                     ]
                                 )
                                 ->andFilterWhere(
                                     [
                                         'like',
                                         'blog_tag_lang.label',
                                         $q,
                                     ]
                                 )
                                 ->limit(20)
                                 ->asArray()
                                 ->all();
            
                if (!empty($categories)) {
                    $out[ 'results' ] = $categories;
                }
            }
        
            return $out;
        }
    
        /**
         * Finds the BlogTag model based on its primary key value.
         * If the model is not found, a 404 HTTP exception will be thrown.
         *
*@param integer $id
         *
         * @return Tag the loaded model
         * @throws NotFoundHttpException if the model cannot be found
         */
        protected function findModel($id)
        {
            if (( $model = Tag::findOne($id) ) !== null) {
                return $model;
            } else {
                throw new NotFoundHttpException('The requested page does not exist.');
            }
        }
    }
